"""
builtin_comp.py - Completion builtins
"""

from _devbuild.gen.runtime_asdl import value_e
from core import completion
from core import error
from core import ui
from core import vm
#from core.pyerror import log
from frontend import flag_spec
from frontend import args
from frontend import consts
from frontend import lexer_def
from frontend import option_def
from core import state

from typing import Dict, List, Union, Iterator, TYPE_CHECKING
if TYPE_CHECKING:
  from _devbuild.gen.runtime_asdl import cmd_value__Argv
  from core.completion import Lookup, OptionState, Api, UserSpec
  from core.ui import ErrorFormatter
  from core.state import Mem
  from frontend.args import _Attributes
  from frontend.parse_lib import ParseContext
  from osh.cmd_eval import CommandEvaluator
  from osh.split import SplitContext
  from osh.word_eval import NormalWordEvaluator

from mycpp import mylib
if mylib.PYTHON:
  # Hack because we don't want libcmark.so dependency for build/dev.sh minimal
  try:
    from _devbuild.gen import help_
  except ImportError:
    class _DummyModule(object): pass
    help_ = _DummyModule()
    help_.TOPICS = []


def _DefineFlags(spec):
  spec.ShortFlag('-F', args.String, help='Complete with this function')
  spec.ShortFlag('-W', args.String, help='Complete with these words')
  spec.ShortFlag('-P', args.String,
      help='Prefix is added at the beginning of each possible completion after '
           'all other options have been applied.')
  spec.ShortFlag('-S', args.String,
      help='Suffix is appended to each possible completion after '
           'all other options have been applied.')
  spec.ShortFlag('-X', args.String,
      help='''
A glob pattern to further filter the matches.  It is applied to the list of
possible completions generated by the preceding options and arguments, and each
completion matching filterpat is removed from the list. A leading ! in
filterpat negates the pattern; in this case, any completion not matching
filterpat is removed. 
''')


def _DefineOptions(spec):
  """Common -o options for complete and compgen."""
  spec.InitOptions()

  # bashdefault, default, filenames, nospace are used in git
  spec.Option(None, 'bashdefault',
      help='If nothing matches, perform default bash completions')
  spec.Option(None, 'default',
      help="If nothing matches, use readline's default filename completion")
  spec.Option(None, 'filenames',
      help="The completion function generates filenames and should be "
           "post-processed")
  spec.Option(None, 'dirnames',
      help="If nothing matches, perform directory name completion")
  spec.Option(None, 'nospace',
      help="Don't append a space to words completed at the end of the line")
  spec.Option(None, 'plusdirs',
      help="After processing the compspec, attempt directory name completion "
      "and return those matches.")


def _DefineActions(spec):
  """Common -A actions for complete and compgen."""

  # NOTE: git-completion.bash uses -f and -v. 
  # My ~/.bashrc on Ubuntu uses -d, -u, -j, -v, -a, -c, -b
  spec.InitActions()
  spec.Action('a', 'alias')
  spec.Action('b', 'binding')
  spec.Action('c', 'command')
  spec.Action('d', 'directory')
  spec.Action('f', 'file')
  spec.Action('j', 'job')
  spec.Action('u', 'user')
  spec.Action('v', 'variable')
  spec.Action(None, 'function')
  spec.Action(None, 'helptopic')  # help
  spec.Action(None, 'setopt')  # set -o
  spec.Action(None, 'shopt')  # shopt -s
  spec.Action(None, 'signal')  # kill -s
  spec.Action(None, 'stopped')


class _FixedWordsAction(completion.CompletionAction):
  def __init__(self, d):
    # type: (Union[Dict, List[str]]) -> None
    self.d = d

  def Matches(self, comp):
    # type: (Api) -> Iterator[Union[Iterator, Iterator[str]]]
    for name in sorted(self.d):
      if name.startswith(comp.to_complete):
        yield name


class SpecBuilder(object):

  def __init__(self,
               cmd_ev,  # type: CommandEvaluator
               parse_ctx,  # type: ParseContext
               word_ev,  # type: NormalWordEvaluator
               splitter,  # type: SplitContext
               comp_lookup,  # type: Lookup
               ):
    # type: (...) -> None
    """
    Args:
      cmd_ev: CommandEvaluator for compgen -F
      parse_ctx, word_ev, splitter: for compgen -W
    """
    self.cmd_ev = cmd_ev
    self.parse_ctx = parse_ctx
    self.word_ev = word_ev
    self.splitter = splitter
    self.comp_lookup = comp_lookup

  def Build(self, argv, arg, base_opts):
    # type: (List[str], _Attributes, Dict) -> UserSpec
    """Given flags to complete/compgen, return a UserSpec."""
    cmd_ev = self.cmd_ev

    actions = []

    # NOTE: bash doesn't actually check the name until completion time, but
    # obviously it's better to check here.
    if arg.F:
      func_name = arg.F
      func = cmd_ev.procs.get(func_name)
      if func is None:
        raise error.Usage('Function %r not found' % func_name)
      actions.append(completion.ShellFuncAction(cmd_ev, func, self.comp_lookup))

    # NOTE: We need completion for -A action itself!!!  bash seems to have it.
    for name in arg.actions:
      if name == 'alias':
        a = _FixedWordsAction(self.parse_ctx.aliases)

      elif name == 'binding':
        # TODO: Where do we get this from?
        a = _FixedWordsAction(['vi-delete'])

      elif name == 'command':
        # compgen -A command in bash is SIX things: aliases, builtins,
        # functions, keywords, external commands relative to the current
        # directory, and external commands in $PATH.

        actions.append(_FixedWordsAction(consts.BUILTIN_NAMES))
        actions.append(_FixedWordsAction(self.parse_ctx.aliases))
        actions.append(_FixedWordsAction(cmd_ev.procs))
        actions.append(_FixedWordsAction(lexer_def.OSH_KEYWORD_NAMES))
        actions.append(completion.FileSystemAction(exec_only=True))

        # Look on the file system.
        a = completion.ExternalCommandAction(cmd_ev.mem)

      elif name == 'directory':
        a = completion.FileSystemAction(dirs_only=True)

      elif name == 'file':
        a = completion.FileSystemAction()

      elif name == 'function':
        a = _FixedWordsAction(cmd_ev.procs)

      elif name == 'job':
        a = _FixedWordsAction(['jobs-not-implemented'])

      elif name == 'user':
        a = completion.UsersAction()

      elif name == 'variable':
        a = completion.VariablesAction(cmd_ev.mem)

      elif name == 'helptopic':
        # Note: it would be nice to have 'helpgroup' for help -i too
        a = _FixedWordsAction(help_.TOPICS)

      elif name == 'setopt':
        names = [opt.name for opt in option_def.All() if opt.builtin == 'set']
        a = _FixedWordsAction(names)

      elif name == 'shopt':
        names = [opt.name for opt in option_def.All() if opt.builtin == 'shopt']
        a = _FixedWordsAction(names)

      elif name == 'signal':
        a = _FixedWordsAction(['TODO:signals'])

      elif name == 'stopped':
        a = _FixedWordsAction(['jobs-not-implemented'])

      else:
        raise NotImplementedError(name)

      actions.append(a)

    # e.g. -W comes after -A directory
    if arg.W is not None:  # could be ''
      # NOTES:
      # - Parsing is done at REGISTRATION time, but execution and splitting is
      #   done at COMPLETION time (when the user hits tab).  So parse errors
      #   happen early.
      w_parser = self.parse_ctx.MakeWordParserForPlugin(arg.W)

      arena = self.parse_ctx.arena
      try:
        arg_word = w_parser.ReadForPlugin()
      except error.Parse as e:
        ui.PrettyPrintError(e, arena)
        raise  # Let 'complete' or 'compgen' return 2

      a = completion.DynamicWordsAction(
          self.word_ev, self.splitter, arg_word, arena)
      actions.append(a)

    extra_actions = []
    if base_opts.get('plusdirs'):
      extra_actions.append(completion.FileSystemAction(dirs_only=True))

    # These only happen if there were zero shown.
    else_actions = []
    if base_opts.get('default'):
      else_actions.append(completion.FileSystemAction())
    if base_opts.get('dirnames'):
      else_actions.append(completion.FileSystemAction(dirs_only=True))

    if not actions and not else_actions:
      raise error.Usage('No actions defined in completion: %s' % argv)

    p = completion.DefaultPredicate
    if arg.X:
      filter_pat = arg.X
      if filter_pat.startswith('!'):
        p = completion.GlobPredicate(False, filter_pat[1:])
      else:
        p = completion.GlobPredicate(True, filter_pat)
    return completion.UserSpec(actions, extra_actions, else_actions, p,
                               prefix=arg.P or '', suffix=arg.S or '')


if mylib.PYTHON:
  # git-completion.sh uses complete -o and complete -F
  COMPLETE_SPEC = flag_spec.FlagSpecAndMore('complete')

  _DefineFlags(COMPLETE_SPEC)
  _DefineOptions(COMPLETE_SPEC)
  _DefineActions(COMPLETE_SPEC)

  COMPLETE_SPEC.ShortFlag('-E',
      help='Define the compspec for an empty line')
  COMPLETE_SPEC.ShortFlag('-D',
      help='Define the compspec that applies when nothing else matches')


class Complete(vm._Builtin):
  """complete builtin - register a completion function.

  NOTE: It's has an CommandEvaluator because it creates a ShellFuncAction, which
  needs an CommandEvaluator.
  """
  def __init__(self, spec_builder, comp_lookup):
    # type: (SpecBuilder, Lookup) -> None
    self.spec_builder = spec_builder
    self.comp_lookup = comp_lookup

  def Run(self, cmd_val):
    # type: (cmd_value__Argv) -> int
    argv = cmd_val.argv[1:]
    arg_r = args.Reader(argv)
    arg = COMPLETE_SPEC.Parse(arg_r)
    # TODO: process arg.opt_changes
    #log('arg %s', arg)

    commands = arg_r.Rest()

    if arg.D:
      commands.append('__fallback')  # if the command doesn't match anything
    if arg.E:
      commands.append('__first')  # empty line

    if not commands:
      self.comp_lookup.PrintSpecs()
      return 0

    base_opts = dict(arg.opt_changes)
    try:
      user_spec = self.spec_builder.Build(argv, arg, base_opts)
    except error.Parse as e:
      # error printed above
      return 2
    for command in commands:
      self.comp_lookup.RegisterName(command, base_opts, user_spec)

    patterns = []
    for pat in patterns:
      self.comp_lookup.RegisterGlob(pat, base_opts, user_spec)

    return 0


if mylib.PYTHON:
  COMPGEN_SPEC = flag_spec.FlagSpecAndMore('compgen')  # for -o and -A

  # TODO: Add -l for COMP_LINE.  -p for COMP_POINT ?
  _DefineFlags(COMPGEN_SPEC)
  _DefineOptions(COMPGEN_SPEC)
  _DefineActions(COMPGEN_SPEC)


class CompGen(vm._Builtin):
  """Print completions on stdout."""

  def __init__(self, spec_builder):
    # type: (SpecBuilder) -> None
    self.spec_builder = spec_builder

  def Run(self, cmd_val):
    argv = cmd_val.argv[1:]
    arg_r = args.Reader(argv)
    arg = COMPGEN_SPEC.Parse(arg_r)

    if arg_r.AtEnd():
      to_complete = ''
    else:
      to_complete = arg_r.Peek()
      arg_r.Next()
      # bash allows extra arguments here.
      #if not arg_r.AtEnd():
      #  raise error.Usage('Extra arguments')

    matched = False

    base_opts = dict(arg.opt_changes)
    try:
      user_spec = self.spec_builder.Build(argv, arg, base_opts)
    except error.Parse as e:
      # error printed above
      return 2

    # NOTE: Matching bash in passing dummy values for COMP_WORDS and COMP_CWORD,
    # and also showing ALL COMPREPLY reuslts, not just the ones that start with
    # the word to complete.
    matched = False 
    comp = completion.Api()
    comp.Update(first='compgen', to_complete=to_complete, prev='', index=-1)
    try:
      for m, _ in user_spec.Matches(comp):
        matched = True
        print(m)
    except error.FatalRuntime:
      # - DynamicWordsAction: We already printed an error, so return failure.
      return 1

    # - ShellFuncAction: We do NOT get FatalRuntimeError.  We printed an error
    # in the executor, but RunFuncForCompletion swallows failures.  See test
    # case in builtin-completion.test.sh.

    # TODO:
    # - need to dedupe results.

    return 0 if matched else 1


if mylib.PYTHON:
  COMPOPT_SPEC = flag_spec.FlagSpecAndMore('compopt')  # for -o
  _DefineOptions(COMPOPT_SPEC)


class CompOpt(vm._Builtin):
  """Adjust options inside user-defined completion functions."""

  def __init__(self, comp_state, errfmt):
    # type: (OptionState, ErrorFormatter) -> None
    self.comp_state = comp_state
    self.errfmt = errfmt

  def Run(self, cmd_val):
    argv = cmd_val.argv[1:]
    arg_r = args.Reader(argv)
    arg = COMPOPT_SPEC.Parse(arg_r)

    if not self.comp_state.currently_completing:  # bash also checks this.
      self.errfmt.Print('compopt: not currently executing a completion function')
      return 1

    self.comp_state.dynamic_opts.update(arg.opt_changes)
    #log('compopt: %s', arg)
    #log('compopt %s', base_opts)
    return 0


if mylib.PYTHON:
  COMPADJUST_SPEC = flag_spec.FlagSpecAndMore('compadjust')

  COMPADJUST_SPEC.ShortFlag('-n', args.String,
      help='Do NOT split by these characters.  It omits them from COMP_WORDBREAKS.')
  COMPADJUST_SPEC.ShortFlag('-s',
      help='Treat --foo=bar and --foo bar the same way.')


class CompAdjust(vm._Builtin):
  """
  Uses COMP_ARGV and flags produce the 'words' array.  Also sets $cur, $prev,
  $cword, and $split.

  Note that we do not use COMP_WORDS, which already has splitting applied.
  bash-completion does a hack to undo or "reassemble" words after erroneous
  splitting.
  """
  def __init__(self, mem):
    # type: (Mem) -> None
    self.mem = mem

  def Run(self, cmd_val):
    # type: (cmd_value__Argv) -> int
    argv = cmd_val.argv[1:]
    arg_r = args.Reader(argv)
    arg = COMPADJUST_SPEC.Parse(arg_r)
    var_names = arg_r.Rest()  # Output variables to set
    for name in var_names:
      # Ironically we could complete these
      if name not in ['cur', 'prev', 'words', 'cword']:
        raise error.Usage('Invalid output variable name %r' % name)
    #print(arg)

    # TODO: How does the user test a completion function programmatically?  Set
    # COMP_ARGV?
    val = self.mem.GetVar('COMP_ARGV')
    if val.tag != value_e.MaybeStrArray:
      raise error.Usage("COMP_ARGV should be an array")
    comp_argv = val.strs

    # These are the ones from COMP_WORDBREAKS that we care about.  The rest occur
    # "outside" of words.
    break_chars = [':', '=']
    if arg.s:  # implied
      break_chars.remove('=')
    # NOTE: The syntax is -n := and not -n : -n =.
    omit_chars = arg.n or ''
    for c in omit_chars:
      if c in break_chars:
        break_chars.remove(c)

    # argv adjusted according to 'break_chars'.
    adjusted_argv = []
    for a in comp_argv:
      completion.AdjustArg(a, break_chars, adjusted_argv)

    if 'words' in var_names:
      state.SetArrayDynamic(self.mem, 'words', adjusted_argv)

    n = len(adjusted_argv)
    cur = adjusted_argv[-1]
    prev = '' if n < 2 else adjusted_argv[-2]

    if arg.s:
      if cur.startswith('--') and '=' in cur:  # Split into flag name and value
        prev, cur = cur.split('=', 1)
        split = 'true'
      else:
        split = 'false'
      # Do NOT set 'split' without -s.  Caller might not have declared it.
      # Also does not respect var_names, because we don't need it.
      state.SetStringDynamic(self.mem, 'split', split)

    if 'cur' in var_names:
      state.SetStringDynamic(self.mem, 'cur', cur)
    if 'prev' in var_names:
      state.SetStringDynamic(self.mem, 'prev', prev)
    if 'cword' in var_names:
      # Same weird invariant after adjustment
      state.SetStringDynamic(self.mem, 'cword', str(n-1))

    return 0
