#!/usr/bin/env python2
"""
build/graph.py

oil-native build.

_build/   # input source
  cpp/
    osh_eval.{h,cc}

_bin/   # output binaries
  gcc/
    osh_eval.{asan,dbg,opt}
    osh_eval.{opt.stripped,symbols}
  clang/
    osh_eval.{asan,dbg,opt,...}
    osh_eval.{opt.stripped,symbols}
"""

from __future__ import print_function

import os
import sys

sys.path.append('vendor')
import ninja_syntax


def log(msg, *args):
  if args:
    msg = msg % args
  print(msg, file=sys.stderr)


def main(argv):
  n = ninja_syntax.Writer(open('build.ninja', 'w'))

  n.comment('Build oil-native')
  n.comment('Generated by %s.' % os.path.basename(__file__))
  n.newline()

  n.rule('ninja-compile',
         command='build/native-steps.sh ninja-compile $in $out',
         description='ninja-compile $in $out')
  n.newline()
  # 1 input and 2 outputs
  n.rule('strip',
         command='build/native-steps.sh strip_ $in $out',
         description='strip $in $out')
  n.newline()

  phony = {
      'osh-eval': [],
      'strip': [],
  }

  for variant in [
      'dbg', 'opt', 'asan', 'alloclog', 'malloc', 'uftrace',
      # leave out tcmalloc since it requires system libs to be installed
      # 'tcmalloc'
      ]:
    b = '_bin/osh_eval.%s' % variant
    n.build(
        b, 'ninja-compile',
        # TODO: move the whole runtime calculation here
        ['_build/cpp/osh_eval.cc'],
        variables=[
            ('variant', variant),
        ])
    n.newline()
    phony['osh-eval'].append(b)

    if variant == 'opt':
      stripped = b + '.stripped'
      symbols = b + '.symbols'
      n.build([stripped, symbols], 'strip', [b],
              variables=[('variant', variant)])
      n.newline()
      phony['strip'].append(stripped)

  #
  # Write phony rules we accumulated
  #
  phony_real = []
  for name in sorted(phony):
    deps = phony[name]
    if deps:
      n.build([name], 'phony', deps)
      n.newline()

      phony_real.append(name)

  n.default(['osh-eval'])

  # All groups
  n.build(['all'], 'phony', phony_real)


if __name__ == '__main__':
  try:
    main(sys.argv)
  except RuntimeError as e:
    print('FATAL: %s' % e, file=sys.stderr)
    sys.exit(1)
